from parallels.common import MigrationError
from parallels.common import version_tuple
from parallels.plesk_api import operator as plesk_ops
from parallels.plesk_api.core import PleskError, PleskErrorCode

def get_plesk_version(plesk_api):
	result = plesk_api.send(
		plesk_ops.server.ServerOperator.Get([
			plesk_ops.server.ServerOperator.Dataset.STAT,
		])
	)
	return result.data.statistics.versions.plesk_version

def get_subscription_sysuser_name(plesk_api, subscription_name):
	hosting = plesk_api.send(plesk_ops.SubscriptionOperator.Get(
		filter=plesk_ops.SubscriptionOperator.FilterByName([subscription_name]),
		dataset=[plesk_ops.SubscriptionOperator.Dataset.HOSTING]
	))[0].data[1].hosting
	if isinstance(hosting, plesk_ops.SubscriptionHostingVirtual):
		return hosting.properties.get('ftp_login')
	else:
		return None

def list_mail_users(plesk_api, domain_name):
	try:
		site_id = plesk_api.send(
			plesk_ops.SiteOperator.Get(
				dataset=[plesk_ops.SiteOperator.Dataset.GEN_INFO],
				filter=plesk_ops.SiteOperator.FilterByName([domain_name])
			)
		)[0].data[0]
	except PleskError as e:
		# webspace with such name does not exist - simply return empty list
		if e.code == PleskErrorCode.OBJECT_DOES_NOT_EXIST: 
			return []
		else:
			raise

	api_response = plesk_api.send(
		plesk_ops.MailOperator.Get(
			mailbox=[plesk_ops.MailOperator.Dataset.GEN_INFO],
			filter=plesk_ops.MailOperator.FilterBySiteId([site_id])
		)
	)

	list_mail_users = []
	for mail_response in api_response:
		list_mail_users.append(mail_response.data.name)

	return list_mail_users

def list_databases(plesk_api, domain_name, database_type):
	"""List databases registered in panel for specified domain
	
	Arguments:
	- plesk_api - Plesk API client, instance of
	  parallels.plesk_api.client.Client
	- domain_name - name of domain to list databases
	- database_type - type of databases to filter ('mysql' or 'mssql' or
	  'postgresql')

	Returns:
		list of tuples, first element - database name, second element - 
		list of related database user logins
	"""
	try:
		operator = plesk_ops.DatabaseOperator.GetDb(
			filter=plesk_ops.DatabaseOperator.GetDb.FilterByWebspaceName([
				domain_name
			])
		)
		api_response = plesk_api.send(operator)

		db_list = []
		for database_api_response in api_response:
			if database_type == database_api_response.data.type:
				users = _list_database_users(
					plesk_api, database_api_response.data.id)
				db_list.append((database_api_response.data.name, users))
		return db_list
	except PleskError as e:
		# webspace with such name does not exist, or there are no databases  -
		# simply return empty list
		if e.code in (
			PleskErrorCode.OBJECT_DOES_NOT_EXIST, 
			PleskErrorCode.OWNER_DOES_NOT_EXIST
		): 
			return []
		else:
			raise

def change_dedicated_iis_app_pool_state(plesk_api, subscription_name, state):
	"""Switch on or off dedicated application pool on subscription

	Arguments:
	- plesk_api - Plesk API client, instance of
	  parallels.plesk_api.client.Client
	- subscription_name - name of Windows-based subscription to change state of
	  dedicated application pool on
	- state - boolean, True to switch application pool on, False to switch
	  application off
	"""
	result = plesk_api.send(
		plesk_ops.SubscriptionOperator.Set(
			filter=plesk_ops.SubscriptionOperator.FilterByName([
				subscription_name
			]),
			hosting=plesk_ops.SubscriptionHostingVirtual(
				properties=[
					('iis_app_pool', 'true' if state else 'false'),
				],
				ip_addresses=None # do not change
			),
			limits=None,
		)
	)

	if len(result) != 1 or not result[0].ok:
		raise MigrationError(
			u"Failed to %s IIS dedicated application pool, "
			"see debug log for more details" % (
				'enable' if state else 'disable'
			)
		)

def is_subscription_suspended(plesk_api, subscription_name):
	if version_tuple(plesk_api.api_version) <= (1,6,0,2):
		if version_tuple(plesk_api.api_version) < (1,6,0,0):
			filter = plesk_ops.DomainOperator.FilterByName8([subscription_name])
		else:
			filter = plesk_ops.DomainOperator.FilterByDomainName([subscription_name])
		return plesk_api.send(
			plesk_ops.DomainOperator.Get(
				filter=filter,
				dataset=[plesk_ops.DomainOperator.Dataset.GEN_INFO]
			)
		)[0].data[1].gen_info.status != 0
	else:
		return plesk_api.send(plesk_ops.SubscriptionOperator.Get(
			filter=plesk_ops.SubscriptionOperator.FilterByName([subscription_name]),
			dataset=[plesk_ops.SubscriptionOperator.Dataset.GEN_INFO]
		))[0].data[1].gen_info.status != 0

def _list_database_users(plesk_api, database_id):
	operator = plesk_ops.DatabaseOperator.GetDbUsers(
		filter=plesk_ops.DatabaseOperator.GetDbUsers.FilterByDbIds([
			database_id
		])
	)
	api_response = plesk_api.send(operator)
	return [
		user_api_response.data.login
		for user_api_response in api_response
	]

