## Copyright (C) 2001 Paul Kienzle <pkienzle@users.sf.net>
##
## This program is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING. If not, see
## <https://www.gnu.org/licenses/>.

## -*- texinfo -*-
## @deftypefn {Function File} {} dctmtx (@var{n})
## Return the DCT transformation matrix of size @var{n}-by-@var{n}.
##
## If A is an @var{n}-by-@var{n} matrix, then the following are true:
##
## @example
## @group
##     T*A    == dct(A),  T'*A   == idct(A)
##     T*A*T' == dct2(A), T'*A*T == idct2(A)
## @end group
## @end example
##
## A DCT transformation matrix is useful for doing things like jpeg
## image compression, in which an 8x8 DCT matrix is applied to
## non-overlapping blocks throughout an image and only a subblock on the
## top left of each block is kept.  During restoration, the remainder of
## the block is filled with zeros and the inverse transform is applied
## to the block.
##
## @seealso{dct, idct, dct2, idct2}
## @end deftypefn

function T = dctmtx(n)

  if nargin != 1
    print_usage;
  endif

  if n > 1
    T = [ sqrt(1/n)*ones(1,n) ; ...
          sqrt(2/n)*cos((pi/2/n)*([1:n-1]'*[1:2:2*n])) ];
  elseif n == 1
    T = 1;
  else
    error ("dctmtx: n must be at least 1");
  endif

endfunction
